
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/leds.h>

#include "dvlled_mod.h"

#define MODULE_NAME  "dvlled"
MODULE_LICENSE("GPL");

//
// Prototypes
//
static int dvlled_probe(struct platform_device *pdev);
static int dvlled_remove(struct platform_device *pdev);


//
// Device and driver structures
//

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,31)
//older kernel
struct platform_device dvl_platform_device =
{
	.name              = LED_DEVICE_NAME,
	.id                = -1,
	.dev.platform_data = &global_leds,
	.num_resources     = 0,
	.resource          = NULL,
};
#else
//newer kernel
struct platform_device *dvl_platform_device;
#endif

struct platform_driver dvl_platform_driver =
{
	.probe  = dvlled_probe,
	.remove = dvlled_remove,
	.driver =
	{
		.name = LED_DEVICE_NAME,
	},
};


static struct dvlled_device *led_devices = NULL;

//
// Helper functions
//

int num_leds(void)
{
	int rv = 0;
	struct dvlled_entry *l;
	
	for (l = global_leds, rv = 0; l->type != TYPE_NONE; l++, rv++)
	{
	}

	return rv;
}

int getGpioByName(struct led_classdev *led, const char* name)
{
	struct dvlled_entry *dvlLedConfig;

    int i;
    //make dvl led config accessable
	dvlLedConfig = (struct dvlled_entry*)&global_leds;

    //configure mirrored gpio and inverted state
	for (i = 0; i < num_leds(); i++)
	{
        //Note: parameter led contains the current led
        //      dvlLedConfig the config of all dvl known leds

        //skip all not desired leds
        if (strcmp(dvlLedConfig[i].name,led->name))
        {
            //printk("skip %s, this is %s\n",dvlLedConfig[i].name,led->name);
            continue;
        }

        return dvlLedConfig[i].line1;
    }

    return 0;
}

#ifdef _DVL_USE_GENERIC_LED_TRIGGER  
/* checks if a trigger is already active on this led
 * returns the trigger state or -1 if no led with this name was found
 */
int isTriggerActiveOnLEDByLedName(const char* name)
{
    int i = 0;

    while (blink_table[i].led_type != TYPE_NONE)
    {
        if (strcmp(name, blink_table[i].led_name) == 0)
        {
            return blink_table[i].is_activated;
        }
        i++;
    }
    return -1;
}

/* sets the active trigger flag or a led to the new value
 * returns the set value or -1 if no led with this name was found
 */
int setActivateTriggerOnLEDByLedName(const char* name, int activate)
{
    int i = 0;

    while (blink_table[i].led_type != TYPE_NONE)
    {
        if (strcmp(name, blink_table[i].led_name) == 0)
        {
            blink_table[i].is_activated = activate;

            return blink_table[i].is_activated;
        }
        i++;
    }

    return -1;
}

/* gives back the blink states for a LED
 * returns the pointer to the state table or NULL if none was found
 */
struct led_blink_state* getLedStatesByLedName(const char* name)
{
    int i = 0;

    while (blink_table[i].led_type != TYPE_NONE)
    {
        if (strcmp(name, blink_table[i].led_name) == 0)
        {
            return blink_table[i].led_states;
        }
        i++;
    }

    return NULL;
}
#endif

int getGpioPolByName(struct led_classdev *led, const char* name)
{
	struct dvlled_entry *dvlLedConfig;

    int i;
    //make dvl led config accessable
	dvlLedConfig = (struct dvlled_entry*)&global_leds;

    //configure mirrored gpio and inverted state
	for (i = 0; i < num_leds(); i++)
	{
        //Note: parameter led contains the current led
        //      dvlLedConfig the config of all dvl known leds

        //skip all not desired leds
        if (strcmp(dvlLedConfig[i].name,led->name))
        {
            //printk("skip %s, this is %s\n",dvlLedConfig[i].name,led->name);
            continue;
        }

        return dvlLedConfig[i].polarity;
    }

    return POL_HA;
}

static void led_set_value(int index, int value)
{
	struct dvlled_entry *l = &(global_leds[index]);
	
	int value_on  = (l->polarity == POL_HA) ? 1 : 0;
	int value_off = 1 - value_on;
	
	if (l->type == TYPE_LED || l->type == TYPE_MIRRORED_LED)
	{
    	if (l->line1 >= 0)
		{
			if (value != 0)
				dvl_led_set(l->line1, value_on);
			else
				dvl_led_set(l->line1, value_off);
		}
	}
	else if (l->type == TYPE_DUAL_LED)
	{
		if (l->line1 >= 0)
		{
			if ((value & 1) != 0)
				dvl_led_set(l->line1, value_on);
			else
				dvl_led_set(l->line1, value_off);
		}
		
		if (l->line2 >= 0)
		{
			if ((value & 2) != 0)
				dvl_led_set(l->line2, value_on);
			else
				dvl_led_set(l->line2, value_off);
		}
	}
}

//
// Device callbacks
//

static int dvlled_probe(struct platform_device *pdev)
{
	int num = num_leds();
	int i;

	dvl_led_probe(pdev);

	led_devices = kzalloc(num * sizeof(struct dvlled_device), GFP_KERNEL);

	for (i = 0; i < num; i++)
	{
		struct dvlled_entry *l = &(global_leds[i]);
		int rc;

		snprintf(led_devices[i].name, sizeof(led_devices[i].name), "%s", global_leds[i].name);
		led_devices[i].index = i;

		led_devices[i].ancestor.name = led_devices[i].name;
		led_devices[i].ancestor.flags = 0;
		led_devices[i].ancestor.brightness = LED_OFF;
		led_devices[i].ancestor.brightness_set = dvlled_brightness_set;
        led_devices[i].ancestor.trigger_data = NULL;

		rc = led_classdev_register(&pdev->dev, &led_devices[i].ancestor);
		if (rc < 0)
		{
			led_devices[i].ancestor.name = 0;
			dvlled_remove(pdev);
			return rc;
		}

        //Note: a mirrored led use line2 to store the mirrored gpio, refer ledtrig-gpio
		if (l->type == TYPE_LED || l->type == TYPE_MIRRORED_LED)
		{
			if (global_leds[i].line1 >= 0)
			{
			    led_set_value(global_leds[i].line1, 0); //deactivate output (means: HA -> 0, LA -> 1)
				dvl_led_config_out(global_leds[i].line1);
				led_set_value(global_leds[i].line1, 0); //deactivate output again to make sure, leds are off
			}
		}
		else if (l->type == TYPE_DUAL_LED)
		{
			if (global_leds[i].line1 >= 0)
				dvl_led_config_out(global_leds[i].line1);
			if (global_leds[i].line2 >= 0)
				dvl_led_config_out(global_leds[i].line2);
		}

    }
	return 0;
}

static int dvlled_remove(struct platform_device *pdev)
{
	int num = num_leds();
	int i;

	for(i = 0; i < num; i++)
	{
		if (led_devices[i].ancestor.name != 0)
		{
			led_classdev_unregister(&led_devices[i].ancestor);
			led_devices[i].ancestor.name = 0;
		}
	}

	kfree(led_devices);
	led_devices = NULL;

	dvl_led_remove(pdev);

	return 0;
}

void dvlled_brightness_set(struct led_classdev *pled, enum led_brightness value)
{
	const struct dvlled_device *const dev = container_of(pled, struct dvlled_device, ancestor);

	led_set_value(dev->index, (int) value);
}

void set_lan_led(unsigned int on)
{
	if (on)
	{
		led_set_value(lan_led_idx, 1);
	}
	else
	{
		led_set_value(lan_led_idx, 0);
	}
}

EXPORT_SYMBOL(set_lan_led);

void set_wireless_2g_led(unsigned int on)
{
	if (on)
	{
		led_set_value(wireless_2g_led_idx, 1);
	}
	else
	{
		led_set_value(wireless_2g_led_idx, 0);
	}
}

EXPORT_SYMBOL(set_wireless_2g_led);

void set_wireless_5g_led(unsigned int on)
{
	if (on)
	{
		led_set_value(wireless_5g_led_idx, 1);
	}
	else
	{
		led_set_value(wireless_5g_led_idx, 0);
	}
}

EXPORT_SYMBOL(set_wireless_5g_led);

//
// Init/Exit
//

static int __init init_mod(void)
{
	int rc = 0;
	
	dvl_led_init();

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,31)	
    //older kernel
	platform_device_register(&dvl_platform_device);
#else
    //newer kernel => use common interfaces
    //NOTICE:
    //Using platform_device_alloc() and platform_device_add() to register your device;
    //in contrast to platform_device_register this cause no OOPS due to a not
    //defined release function

    //allocate memory for a new platform device
    dvl_platform_device = platform_device_alloc(LED_DEVICE_NAME,-1);

    //TODO: add missing led/gpio data
    //      platform_device_add_data() does not fit, because global_leds is a dynamical sized
    //      array[]. 
    //platform_device_add_data(dvl_platform_device, &global_leds, sizeof global_leds);
    dvl_platform_device->dev.platform_data = &global_leds;

    //no resources are available
    //platform_device_add_resources(dvl_platform_device, 0, 0);

    //register device
    rc = platform_device_add(dvl_platform_device);
    if (rc < 0) 
    {
		printk(KERN_INFO "Module %s could not add platform device\n", MODULE_NAME);
        return rc;
    }
#endif
	rc = platform_driver_register(&dvl_platform_driver);
	if (rc < 0)
    {
		printk(KERN_INFO "Module %s could not register platform driver\n", MODULE_NAME);
        return rc;
    }

#ifdef _DVL_USE_LED_TRIGGER_GPIO   
    rc = led_trigger_register(&gpio_led_trigger);
    if (rc < 0)
    {
        printk("%s: can not register trigger %s\n",MODULE_NAME,gpio_led_trigger.name);
        return rc;
    }
#endif

#ifdef _DVL_USE_LED_TRIGGER_WIFI
    rc = led_trigger_register(&wifiled_trigger);
    if (rc < 0)
    {
        printk("%s: can not register trigger %s\n",MODULE_NAME,wifiled_trigger.name);
        return rc;
    }
#endif

#ifdef _DVL_USE_GENERIC_LED_TRIGGER   
    rc = led_trigger_register(&dvl_generic_trigger);
    if (rc < 0)
    {
        printk("%s: can not register trigger %s\n",MODULE_NAME,dvl_generic_trigger.name);
        return rc;
    }
#endif

	if (rc >= 0)
		printk(KERN_INFO "Module %s initialized\n", MODULE_NAME);   

	return rc;
}

static void __exit exit_mod(void)
{
	platform_driver_unregister(&dvl_platform_driver);

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,31)	
    //older kernel
	platform_device_unregister(&dvl_platform_device);
#else
    //newer kernel
	platform_device_del(dvl_platform_device);
    kfree(dvl_platform_device);
#endif

#ifdef _DVL_USE_LED_TRIGGER_GPIO   
    led_trigger_unregister(&gpio_led_trigger);
#endif
	
#ifdef _DVL_USE_GENERIC_LED_TRIGGER   
    led_trigger_unregister(&dvl_generic_trigger);
#endif
	
	printk(KERN_INFO "Module %s removed\n", MODULE_NAME);
}

MODULE_AUTHOR("Christian Petry");
MODULE_DESCRIPTION("LED module");

module_init(init_mod);
module_exit(exit_mod);
