/*
 * RT3052 driver for LEDs
 *
 * Author: devolo AG (http://www.devolo.de)
 *
 * Copyright (c) 2009 devolo AG
 *
 */

#include <linux/autoconf.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/spinlock.h>
#include <linux/leds.h>
#include <asm/io.h>
#include <asm/rt2880/generic.h>

#include <../drivers/char/ralink_gpio.h>
#include <asm-mips/rt2880/surfboardint.h>


#define GPIO_LED_WPS 14

struct rt3052led_device
{
	struct led_classdev ancestor;
	
	char name[3];
	int pos;
};

static struct rt3052led_device *led_devices;
static struct dvl_rt3052_leds *led_data;

static void dvl_gpio_line_config_out(int line)
{
	unsigned int tmp;

	// polarity
	tmp = le32_to_cpu(*(volatile u32 *)(RALINK_REG_PIOPOL));
	//tmp &= ~(unsigned)(1 << line);
	tmp |= (1 << line);
	*(volatile u32 *)(RALINK_REG_PIOPOL) = cpu_to_le32(tmp);
	
	// direction
	tmp = le32_to_cpu(*(volatile u32 *)(RALINK_REG_PIODIR));
	tmp |= (1 << line);
	*(volatile u32 *)(RALINK_REG_PIODIR) = cpu_to_le32(tmp);
}

static void dvl_gpio_line_set(int line, int value)
{
	unsigned int tmp;
	
	tmp = le32_to_cpu(*(volatile u32 *)(RALINK_REG_PIODATA));
	if (value)
		tmp |= (1 << line);
	else
		tmp &= ~(1 << line);
	*(volatile u32 *)(RALINK_REG_PIODATA) = cpu_to_le32(tmp);
}

static void rt3052led_brightness_set(struct led_classdev *pled, enum led_brightness value)
{
	const struct rt3052led_device *const rt3052_dev = container_of(pled, struct rt3052led_device, ancestor);

	if (value > 0)
		dvl_gpio_line_set(GPIO_LED_WPS, 1);
	else
		dvl_gpio_line_set(GPIO_LED_WPS, 0);
}

static int rt3052led_remove(struct platform_device *pdev)
{
	int i;

	for(i = 0; i < led_data->quantity; i++)
	{
		if (led_devices[i].ancestor.name != 0)
		{
			led_classdev_unregister(&led_devices[i].ancestor);
			led_devices[i].ancestor.name = 0;
		}
	}

	kfree(led_devices);
	led_devices = 0;
	led_data = 0;

	return 0;
}

static int rt3052led_probe(struct platform_device *pdev)
{
	int i, q;

	led_data = (struct dvl_rt3052_leds*)pdev->dev.platform_data;
	q = led_data->quantity;

	if (q <= 0)
		return -1;

	led_devices = kzalloc(q * sizeof(struct rt3052led_device), GFP_KERNEL);

	for (i = 0; i < q; i++)
	{
		int rc;

		led_devices[i].name[0] = 'L';
		led_devices[i].name[1] = '1' + i;
		led_devices[i].name[2] = '\0';
		led_devices[i].pos = 1 << (q - 1 - i);

		led_devices[i].ancestor.name = led_devices[i].name;
		led_devices[i].ancestor.flags = 0;
		led_devices[i].ancestor.brightness = LED_OFF;
		led_devices[i].ancestor.brightness_set = rt3052led_brightness_set;

		rc = led_classdev_register(&pdev->dev, &led_devices[i].ancestor);
		if (rc < 0)
		{
			led_devices[i].ancestor.name = 0;
			rt3052led_remove(pdev);
			return rc;
		}
		
		dvl_gpio_line_config_out(GPIO_LED_WPS);
		dvl_gpio_line_set(GPIO_LED_WPS, 0);
	}

	return 0;
}

static struct platform_driver rt3052led_driver =
{
	.probe   = rt3052led_probe,
	.remove  = rt3052led_remove,
	.driver  = {
		.name = "RT3052-LED",
	},
};

static int __init rt3052led_init(void)
{
	return platform_driver_register(&rt3052led_driver);
}

static void __exit rt3052led_exit(void)
{
	platform_driver_unregister(&rt3052led_driver);
}

module_init(rt3052led_init);
module_exit(rt3052led_exit);

MODULE_AUTHOR("<info@devolo.de>");
MODULE_DESCRIPTION("RT3052 LED driver");
MODULE_LICENSE("Unknown");
